/**
 * @file cheat-panel.ts
 * @copyright 2024, Firaxis Games
 * @description Screen to display pantheon choices after choosing them
 */
import Panel from '/core/ui/panel-support.js';
import { MustGetElement } from '/core/ui/utilities/utilities-dom.js';
import FocusManager from '/core/ui/input/focus-manager.js';
import { InputEngineEventName } from '/core/ui/input/input-support.js';
import NavTray from '/core/ui/navigation-tray/model-navigation-tray.js';

function debugObjFunctions(obj) {
    var res = [];
    for(var m in obj) {
        res.push(m)
        console.error(m);
    }
    console.error(res.join(", "))
    // const props = Object.getOwnPropertyNames(obj) || [];
    // console.error(props.join(", "));
}

class ScreenCheatPanel extends Panel {
    constructor() {
        super(...arguments);
        this.engineInputListener = this.onEngineInput.bind(this);

        this.cheats = [
            {class: ".cheat-panel-money", callback: this.addMoney},
            {class: ".cheat-panel-influence", callback: this.addInfluence},
            {class: ".cheat-panel-unit-restore-movement", callback: this.restoreUnitMovement},
            {class: ".cheat-panel-unit-heal", callback: this.healUnit},
            {class: ".cheat-panel-unit-give-exp", callback: this.unitGiveExp},
            {class: ".cheat-panel-complete-build-queue", callback: this.completeProduction},
            // {class: ".cheat-panel-city-heal", callback: this.cityHeal},
            {class: ".cheat-panel-complete-research", callback: this.completeResearch},
            {class: ".cheat-panel-complete-civic", callback: this.completeCivic},
        ];
    }



    onInitialize() {
        console.log("Cheat Panel Loaded");

        this.pantheonFrame = MustGetElement(".cheat-frame", this.Root);
        this.enableOpenSound = true;
        this.enableCloseSound = true;

    }


    onAttach() {
        this.playAnimateInSound();
        this.Root.addEventListener(InputEngineEventName, this.engineInputListener);
        this.pantheonFrame.addEventListener('subsystem-frame-close', () => { this.close(); });

        for (const cheat of this.cheats) {
            const button = this.Root.querySelector(cheat.class);
            if (button) {
                button.addEventListener("action-activate", cheat.callback.bind(this));
            }
            else {
                console.error(`[Cheat Panel] Unable to find button for ${cheat.class}`);
            }
        }

        const player = Players.get(GameContext.localPlayerID);
        if (!player) {
            console.error("cheat-panel: onAttach() - no local player found!");
            return;
        }
        const playerReligion = player.Religion;
        if (!playerReligion) {
            console.error("cheat-panel: onAttach() - no player religion found!");
            return;
        }
        FocusManager.setFocus(this.Root);
    }
    onDetach() {
        this.Root.removeEventListener(InputEngineEventName, this.engineInputListener);
    }
    onReceiveFocus() {
        super.onReceiveFocus();
        NavTray.clear();
        NavTray.addOrUpdateGenericBack();
    }
    onEngineInput(inputEvent) {
        if (inputEvent.detail.status != InputActionStatuses.FINISH) {
            return;
        }
        if (inputEvent.isCancelInput() || inputEvent.detail.name == 'sys-menu') {
            this.close();
            inputEvent.stopPropagation();
            inputEvent.preventDefault();
        }
    }

    addMoney() {
        console.error("cheat-panel: Money Button");
        const player = Players.get(GameContext.localPlayerID);
        if(player) {
            const treasury = player.Treasury;
            console.error(`[Cheat Panel] Gold Balance: ${treasury.goldBalance}`);
            treasury.changeGoldBalance(1000);
        }
    }

    addInfluence() {
        console.error("cheat-panel: Influence");
        const player = Players.get(GameContext.localPlayerID);
        if(player) {
            const diplomacyTreasury = player.DiplomacyTreasury;
            console.error(`[Cheat Panel] Influence Balance: ${diplomacyTreasury.diplomacyBalance}`);
            diplomacyTreasury.changeDiplomacyBalance(1000);
        }
    }

    restoreUnitMovement() {
        // Thanks https://forums.civfanatics.com/threads/how-to-enable-the-in-game-console.694859/post-16773523
        const id = UI.Player.getHeadSelectedUnit();
        if ((id != null) && (id.owner != -1 && id.id != -1)) {
            const unit = Units.get(id);
            if (unit) {
                
                let typeName = "";
                const unitInfo = GameInfo.Units.lookup(unit.type);
                if (unitInfo != null) {
                    typeName = unitInfo.UnitType.replace("UNIT_", "");
                } else {
                    typeName = unit.type.toString();
                }
                console.warn(typeName);
                
                Units.restoreMovement(unit.id);
            }
        }
    }

    healUnit() {
        // debugObjFunctions(Units);
        // return;
        const id = UI.Player.getHeadSelectedUnit();
        if ((id != null) && (id.owner != -1 && id.id != -1)) {
            const unit = Units.get(id);
            if (unit) {
                
                let typeName = "";
                const unitInfo = GameInfo.Units.lookup(unit.type);
                if (unitInfo != null) {
                    typeName = unitInfo.UnitType.replace("UNIT_", "");
                } else {
                    typeName = unit.type.toString();
                }
                console.warn(typeName);
                if (unit.Health) {
                    Units.setDamage(unit.id, 0);
                }
                else {
                    console.error("[Cheat Panel] Unit Health is undefined!");
                }
            }
            else {
                console.error("[Cheat Panel] Unit is undefined!");
            }
        }
        else {
            console.error("[Cheat Panel] Unit is not Selected!");
        }
    }

    unitGiveExp() {
        const id = UI.Player.getHeadSelectedUnit();
        if ((id != null) && (id.owner != -1 && id.id != -1)) {
            const unit = Units.get(id);
            if (unit) {
                
                let typeName = "";
                const unitInfo = GameInfo.Units.lookup(unit.type);
                if (unitInfo != null) {
                    typeName = unitInfo.UnitType.replace("UNIT_", "");
                } else {
                    typeName = unit.type.toString();
                }
                console.warn(typeName);
                Units.changeExperience(unit.id, 20);
            }
            else {
                console.error("[Cheat Panel] Unit is undefined!");
            }
        }
        else {
            console.error("[Cheat Panel] Unit is not Selected!");
        }
    }

    completeProduction() {
        const cityID = UI.Player.getHeadSelectedCity();
        if (cityID) {
            let city = Cities.get(cityID);
            if (city != null) {
                
                let cityName = city.name;
                cityName = cityName.replace('LOC_CITY_NAME_', '');
                                                    
                let cityStatus = "City";
                if ( city.isTown )
                {
                    cityStatus = "Town";
                }

                console.warn(cityName + " - " + cityStatus);
                city.BuildQueue.completeProduction();       
            }
            else {
                console.error("[Cheat Panel] City is undefined!");
            }
        }
        else {
            console.error("[Cheat Panel] City is not Selected!");
        }
    }

    cityHeal() {
        const cityID = UI.Player.getHeadSelectedCity();
        if (cityID) {
            let city = Cities.get(cityID);
            if (city != null) {
                
                let cityName = city.name;
                cityName = cityName.replace('LOC_CITY_NAME_', '');
                                                    
                let cityStatus = "City";
                if ( city.isTown )
                {
                    cityStatus = "Town";
                }

                console.warn(cityName + " - " + cityStatus);
                console.error(city.Districts.cityCenter.id);
                const playerDistricts = Players.Districts.get(GameContext.localPlayerID);
                debugObjFunctions(playerDistricts);
                // city
            }
            else {
                console.error("[Cheat Panel] City is undefined!");
            }
        }
        else {
            console.error("[Cheat Panel] City is not Selected!");
        }
    }

    completeResearch() {
        const player = Players.get(GameContext.localPlayerID);
        if (player) {
            const techs = player.Techs;
            // debugObjFunctions(Players);
            if (techs) {
                const techTreeType = techs.getTreeType();
                const treeObject = Game.ProgressionTrees.getTree(player.id, techTreeType);
                if (treeObject) {
                    let currentResearchNode = treeObject.nodes[treeObject.activeNodeIndex]?.nodeType;
                    const nodeInfo = GameInfo.ProgressionTreeNodes.lookup(currentResearchNode);
                    if (nodeInfo) {
                        // debugObjFunctions(techs);
                        const techName = nodeInfo?.Name ?? '';
                        const techCost = techs.getNodeCost(currentResearchNode);
                        const playerSciencePerTurn = player.Stats?.getNetYield(YieldTypes.YIELD_SCIENCE) ?? 0;
                        const turnsLeft = techs.getTurnsLeft();
                        // console.error(techCost, playerSciencePerTurn, turnsLeft);
                        let grantedScience;
                        if (playerSciencePerTurn > 0) {
                            grantedScience = (playerSciencePerTurn * turnsLeft) * 1.40; // extra 40% to make sure
                        }
                        else {
                            grantedScience = techCost;
                        }
                        Players.grantYield(player.id, YieldTypes.YIELD_SCIENCE, grantedScience);
                        console.error(`[Cheat Panel] Granted ${grantedScience} Science for tech: ${techName}`);
                    }
                };
            }
        }
    }

    completeCivic() {
        const player = Players.get(GameContext.localPlayerID);
        if (player) {
            const culture = player.Culture;
            if (culture) {
                const cultureTreeType = culture.getActiveTree();
                const treeObject = Game.ProgressionTrees.getTree(player.id, cultureTreeType);
                if (treeObject) {
                    let currentCivicNode = treeObject.nodes[treeObject.activeNodeIndex]?.nodeType;
                    const nodeInfo = GameInfo.ProgressionTreeNodes.lookup(currentCivicNode);
                    if (nodeInfo) {
                        const cultureName = nodeInfo?.Name ?? '';
                        const cultureCost = culture.getNodeCost(currentCivicNode);
                        const playerCulturePerTurn = player.Stats?.getNetYield(YieldTypes.YIELD_CULTURE) ?? 0;
                        const turnsLeft = culture.getTurnsLeft();
                        // console.error(cultureCost, playerSciencePerTurn, turnsLeft);
                        let grantedCulture;
                        if (playerCulturePerTurn > 0) {
                            grantedCulture = (playerCulturePerTurn * turnsLeft) * 1.40; // extra 40% to make sure
                        }
                        else {
                            grantedCulture = cultureCost;
                        }
                        Players.grantYield(player.id, YieldTypes.YIELD_CULTURE, grantedCulture);
                        console.error(`[Cheat Panel] Granted ${grantedCulture} Culture for culture: ${cultureName}`);
                    }
                }
            }
        }
    }


}
Controls.define('panel-cheat', {
    createInstance: ScreenCheatPanel,
    description: 'Screen to display Cheats',
    classNames: ['screen-cheat-panel', "top-40", "bottom-2", "left-0", "absolute"],
    content: ['fs://game/base-standard/ui/cheat-panel/cheat-panel.html'],
    attributes: []
});

//# sourceMappingURL=file:///base-standard/ui/cheat-panel/cheat-panel.js.map
